/*
 * gmm.c
 * Tommaso Polonelli
 *
 * Copyright (C) 2016 ETH Zurich, University of Bologna
 * Copyright (C) 2018 Tommaso Polonelli
 *
 * This software may be modified and distributed under the terms
 * of the MIT license.  See the LICENSE file for details.
 *
 * Created on: May 25, 2020
 *
 */

#include "gmm.h"
#include <string.h>

/*********** Constants  ****************/

/* debug */
#define PRINT_IMG	(0)
#define PERF		(1)


/* Operation block size (number of pixel per operators)
 * this impact the L1 memory but decrease number of transfers L1 -> L3
 */
//#define BLOCK_SIZE Width
#define qW 12
#define qM 7
#define qS 7

#define STOREW(x)	FIXEDU2INT(((x) << (qW)))
#define STOREM(x)	FIXEDU2INT(((x) << (qM)))
#define STORES(x)	FIXEDU2INT(((x) << (qS)))

#define GETW(x)		(INT2FIXEDU(x) >> (qW))
#define GETM(x)		(INT2FIXEDU(x) >> (qM))
#define GETS(x)		(INT2FIXEDU(x) >> (qS))


/*------ Global variables  -----------*/



uint8_t SubtractPixel(
	uint8_t pixel, 					// pixel value
	int pIdx,						// pixel index within the tile
	GMM *Gauss, 					// L2 address of the GMM tile
	GMM_Settings * 	GMMconf			// GMM configuration structure

){

	// paramters
	int ImgHeight = GMMconf->ImgHeight;
	int ImgWidth = GMMconf->ImgWidth;
	int ImgSize = GMMconf->ImgSize;
	int MaxModes = GMMconf->MaxModes;

	/* fixed point values */
	Fixedu alpha = GMMconf->alpha;
	Fixedu prune = GMMconf->prune;
	Fixedu one_minus_alpha = GMMconf->one_minus_alpha;
	uint32_t Threshold = GMMconf->Threshold;
	Fixedu variance = GMMconf->variance;
	Fixedu max_variance = GMMconf->max_variance;
	Fixedu min_variance = GMMconf->min_variance;

	Fixedu complexity_prior = GMMconf->complexity_prior;
	Fixedu BG_Th = GMMconf->BG_Th;
	
	//long posLocal;
	int Nmodes;
	Fixedu FlW [MaxModes], FlM [MaxModes], FlS [MaxModes];
	Fixedu tempW, tempM, tempS;

//	for(Block_C = 0;Block_C < BLOCK_SIZE;Block_C++){

	/* Init vars */
	int Fit = 0;
	int BGGaussians = 0;

	Fixedu sumWeights = 0;
	uint8_t thr = 255;
	Fixedu sum = 0;
	Fixedu FixPix = INT2FIXEDU((int)pixel); 


	PRINTF("(%d)-", pixel);

	// Read pixel-wise GMM model
	int modes = 0;
	for (int i = 0; i < MaxModes; i++){
		int gmmIdx = i+pIdx*MaxModes;
		FlW[i] = ((Fixedu)GETW(Gauss[gmmIdx].weight));
		if (FlW[i] > 0) modes++;
		else continue;
		sumWeights += FlW[i];
		FlM[i] = ((Fixedu) GETM(Gauss[gmmIdx].mu))    ;
		PRINTF("%d(%d) ", FIXEDU2INT(FlM[i]), FlW[i]);
		FlS[i] = ((Fixedu)GETS(Gauss[gmmIdx].sigma)) ;
	}

	//Renormalize weights so they sum to 1
	PRINTF("[");
	for (int i = 0; i < modes; i++)	{
		FlW[i] = FIXEDU_DIV(FlW[i],sumWeights);
		PRINTF("%d ", FlW[i]);
	}
	PRINTF("] ");

	// Calculates number of gaussians of the pixel which are part of the BG
	for (int z=0; z<modes; z++){
//		PRINTF("%d ", FlW[z]);
		if (sum < BG_Th){
			sum += FlW[z];
			BGGaussians++;
		}
	}
	PRINTF("-%d(%d)[ ",modes,BGGaussians);


	//BG subtraction and update
	Nmodes = modes;
	for (int i = 0; i < modes; ++i){
		if (!Fit){
			//background distance
			Fixedu distance;
			if(FlM[i]>FixPix) 	distance = (FlM[i]-FixPix);
			else				distance = (FixPix-FlM[i]);
			uint32_t distance2 = (uint32_t) FIXEDU_MUL(distance,distance);
			Fixedu varThr = Threshold * FlS[i];
			PRINTF(" %d(%d-%d)", FIXEDU2INT(distance2),FIXEDU2INT(FlS[i]),FIXEDU2INT(FlM[i]) );

			// update the weight
			FlW[i] = __builtin_pulp_muluRN(one_minus_alpha, FlW[i], QBITS, 1<<(QBITS-1));	//operands are positive and < 1 (Q16.16)
			FlW[i] += -prune;

			// check fit
			if(distance2 < (uint32_t) varThr){
				//the Gaussian belongs to the BG model
				Fit=1;
				// check if this Gaussian is part of the background model
				if(i < BGGaussians)	{
					thr=0;
				}

				//update weight, mean and variance
				FlW[i] += alpha;
				sumWeights += FlW[i];
				//Fixedu k = FIXEDU_DIV(FIXEDU_ONE, FlW[i]);
				Fixedu k = (((uint32_t)(1<<31)) / FlW[i]) <<1;	// 1 / FlW[i]

				//Fixedu k = (Fixedu)((uint32_t)(1<<31) )/ FlW[i];
				FlM[i]  = FIXEDU_MUL_H(one_minus_alpha,FlM[i]) ;
				PRINTF(" F(%d, %d) ",FlW[i],FlM[i] );
				//Fixedu t = FIXEDU_MUL_H(k,alpha)*(uint32_t) pixel;
				Fixedu t = alpha * pixel;
				PRINTF("F(%d-%d-%d)F",FIXEDU2INT(t),t, FIXEDU2INT(k));
				FlM[i] += t;
				PRINTF("F(%d)F", FlM[i]);
				FlS[i]  = FIXEDU_MUL_H(one_minus_alpha,FlS[i]);
				FlS[i] += FIXEDU_MUL_H(distance2,FIXEDU_MUL_H(alpha,k));
				if (FlS[i] > max_variance)
					FlS[i] = max_variance;
				else if (FlS[i] < min_variance)
					FlS[i] = min_variance;
	
				// Sort weights so they are in descending order
				for (int j=i; j > 0; j--){
					if (FlW[j] > FlW[j-1]){
						tempW = FlW[j];
						tempM = FlM[j];
						tempS = FlS[j];
						FlW[j] = FlW[j-1];
						FlM[j] = FlM[j-1];
						FlS[j] = FlS[j-1];
						FlW[j-1] = tempW;
						FlM[j-1] = tempM;
						FlS[j-1] = tempS;
					}
				}
			}else{
				if(FlW[i] < prune){
					PRINTF("FFFF(%d)", FlW[i]);
					FlW[i]= 0;
					Nmodes--;
				}
				sumWeights += FlW[i];
			}
		}else{
			if(FlW[i] < prune){
				PRINTF("FFFF(%d)", FlW[i]);
				FlW[i] = 0;
				Nmodes--;
			}
			sumWeights += FlW[i];
		}
	}

	//Create new mode if match not found
	if (!Fit){

		if (Nmodes == MaxModes){
			FlW[MaxModes-1] = alpha;
			FlM[MaxModes-1] = FixPix;
			FlS[MaxModes-1] = variance;
		}else{
			FlW[Nmodes] = alpha;
			FlM[Nmodes] = FixPix;
			FlS[Nmodes] = variance;
			Nmodes++;
		}
		PRINTF("*%d*",Nmodes);

		if (Nmodes==1){
			FlW[Nmodes-1] = FIXEDU_ONE;
		}else{
			for (int j=Nmodes; j > 0; j--){
				if (FlW[j] > FlW[j-1]){
					tempW = FlW[j];
					tempM = FlM[j];
					tempS = FlS[j];
					FlW[j] = FlW[j-1];
					FlM[j] = FlM[j-1];
					FlS[j] = FlS[j-1];
					FlW[j-1] = tempW;
					FlM[j-1] = tempM;
					FlS[j-1] = tempS;
				}
			}
		}
	}


	//conversion from fixed to floating
	for (int i = 0; i < MaxModes; i++){
		int gmmIdx = pIdx*MaxModes;
		Gauss[i+gmmIdx].weight = (unsigned short)STOREW(FlW[i]);
		Gauss[i+gmmIdx].mu     = (unsigned short)STOREM(FlM[i]);
		Gauss[i+gmmIdx].sigma  = (unsigned short)STORES(FlS[i]);
	}

	PRINTF("] %d\n",thr);

	return thr;

}

